import json


def analyze_over_refuse(file_path):
    """
    Reads a JSONL file and counts the number and percentage of true/false values
    for the 'is_over_refuse' key. If the key is missing, it falls back to
    evaluating the 'min_word_prompt' value.

    Args:
        file_path (str): The path to the input JSONL file.
    """
    # Initialize counters
    total_lines = 0
    processed_count = 0  # Number of lines successfully processed and counted
    true_count = 0
    false_count = 0
    other_values_count = 0  # To record values other than true/false

    try:
        # Safely open the file using a 'with' statement
        with open(file_path, 'r', encoding='utf-8') as infile:
            print(f"Processing file: {file_path}\n")

            for line in infile:
                total_lines += 1
                try:
                    # Parse the current line
                    data = json.loads(line)

                    is_counted = False  # Flag to check if the current line was successfully counted

                    # Check if 'is_over_refuse' key exists
                    if 'is_over_refuse' in data:
                        is_over_refuse_value = data['is_over_refuse']

                        # Update counters based on the key's value
                        if is_over_refuse_value is True:
                            true_count += 1
                            is_counted = True
                        elif is_over_refuse_value is False:
                            false_count += 1
                            is_counted = True
                        else:
                            # If the value is neither True nor False, record it
                            other_values_count += 1
                            print(f"Warning: Non-boolean value found on line {total_lines}: {is_over_refuse_value}")

                    # --- Fallback Logic: Execute if 'is_over_refuse' key is missing ---
                    else:
                        # Check if the fallback key 'min_word_prompt' exists
                        if 'min_word_prompt' in data:
                            min_word_prompt_value = data['min_word_prompt']
                            # If the value is 'NoRefuse', treat is_over_refuse as false
                            if min_word_prompt_value == 'NoRefuse':
                                false_count += 1
                                is_counted = True
                            # Otherwise, treat is_over_refuse as true
                            else:
                                true_count += 1
                                is_counted = True
                        else:
                            # If both keys are missing, the line cannot be processed and is skipped
                            print(f"Warning: Line {total_lines} is missing both 'is_over_refuse' and 'min_word_prompt' keys. Skipped.")

                    if is_counted:
                        processed_count += 1

                except json.JSONDecodeError:
                    print(f"Warning: JSON format error on line {total_lines}. Skipped.")

        # Avoid division by zero error
        if processed_count == 0:
            print("The file is empty or no lines could be processed.")
            return

        # Calculate percentages based on the number of successfully processed lines
        true_percentage = (true_count / processed_count) * 100
        false_percentage = (false_count / processed_count) * 100

        # Print the final statistics
        print("\n--- Analysis Results ---")
        print(f"Total lines in file: {total_lines}")
        print(f"Successfully processed lines: {processed_count}")
        print("-" * 20)
        print(f"Count for 'true':  {true_count} (Percentage: {true_percentage:.2f}%)")
        print(f"Count for 'false': {false_count} (Percentage: {false_percentage:.2f}%)")

        if other_values_count > 0:
            print(f"Count for other values: {other_values_count}")
        print("--- Analysis Complete ---")

    except FileNotFoundError:
        print(f"Error: File not found at: {file_path}")
    except Exception as e:
        print(f"An unexpected error occurred: {e}")


# --- Main execution block ---
if __name__ == "__main__":
    # Define the input file path.
    # Use a raw string (r"...") for the file path to prevent issues with backslashes.
    input_file = r''

    # Call the function to perform the analysis
    analyze_over_refuse(input_file)